/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: socket.c,v 1.12.2.2 2003/04/22 14:13:09 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/socket.h>
#include <zorp/cap.h>
#include <zorp/log.h>
#include <zorp/error.h>

/**
 * z_bind:
 * @fd: socket to bind
 * @addr: address to bind
 *
 * A thin interface around bind() using a ZSockAddr structure for
 * socket address. It enables the NET_BIND_SERVICE capability (should be
 * in the permitted set.
 *
 **/
GIOStatus
z_bind(gint fd, ZSockAddr *addr)
{
  cap_t saved_caps;
  GIOStatus rc;
  
  z_enter();
  saved_caps = cap_save();
  
  cap_enable(CAP_NET_BIND_SERVICE); /* for binding low numbered ports */
  cap_enable(CAP_NET_ADMIN); /* for binding non-local interfaces, transparent proxying */
  
  if (addr->sa_funcs && addr->sa_funcs->sa_bind_prepare)
    addr->sa_funcs->sa_bind_prepare(fd, addr);

  if (addr->sa_funcs && addr->sa_funcs->sa_bind)
    rc = addr->sa_funcs->sa_bind(fd, addr);
  else
    {
      if (addr && z_ll_bind(fd, &addr->sa, addr->salen) < 0)
        {
          /*LOG
            This mesage attempt when low level bind in some reason.
           */
          z_log(NULL, CORE_ERROR, 3, "bind() failed; error='%m'");
          cap_restore(saved_caps);
          return G_IO_STATUS_ERROR;
        }
      rc = G_IO_STATUS_NORMAL;
    }
  cap_restore(saved_caps);
  z_leave();
  return rc;
}


/**
 * z_accept:
 * @fd: accept connection on this socket
 * @newfd: fd of the accepted connection
 * @addr: store the address of the client here
 *
 * Accept a connection on the given fd, returning the newfd and the
 * address of the client in a Zorp SockAddr structure.
 **/
GIOStatus
z_accept(gint fd, gint *newfd, ZSockAddr **addr)
{
  char sabuf[1024];
  int salen = sizeof(sabuf);
  struct sockaddr *sa = (struct sockaddr *) sabuf;
  
  /* NOTE: workaround a Linux 2.4.20 kernel problem */
  sa->sa_family = 999;
  
  do
    {
      *newfd = z_ll_accept(fd, (struct sockaddr *) sabuf, &salen);
    }
  while (*newfd == -1 && z_errno_is(EINTR));
  if (*newfd != -1)
    {
      if (sa->sa_family == 999 && salen == sizeof(sabuf))
        {
          /* workaround the linux 2.4.20 problem */
          sa->sa_family = AF_UNIX;
          salen = 2;
        }
      *addr = z_sockaddr_new((struct sockaddr *) sabuf, salen);
#if ZORPLIB_ENABLE_IPOPTIONS
      z_get_ip_options(newfd, *addr);
#endif
    }
  else if (z_errno_is(EAGAIN))
    {
      return G_IO_STATUS_AGAIN;
    }
  else
    {
      /*LOG
        This message indicates that accepting a connection was failed
        because of the given reason.
       */
      z_log(NULL, CORE_ERROR, 3, "accept() failed; fd='%d', error='%m'", fd);
      return G_IO_STATUS_ERROR;
    }
  return G_IO_STATUS_NORMAL;
}

/**
 * z_connect:
 * @fd: socket to connect
 * @remote: remote address
 *
 * Connect a socket using Zorp style ZSockAddr structure.
 **/
GIOStatus
z_connect(gint fd, ZSockAddr *remote)
{
  int rc;
  
  z_enter();

#if ZORPLIB_ENABLE_IPOPTIONS
  z_set_ip_options(fd, remote);
#endif
  do
    {
      rc = z_ll_connect(fd, &remote->sa, remote->salen);
    }
  while (rc == -1 && z_errno_is(EINTR));
  if (rc == -1)
    {
      if (!z_errno_is(EINPROGRESS))
        {
          int saved_errno = z_errno_get();
      
          /*LOG
            This message indicates that establishing connection failed for
            the given reason.
           */
          z_log(NULL, CORE_ERROR, 3, "connect() failed; fd='%d', error='%m'", fd);
          z_errno_set(saved_errno);
        }
      z_leave();
      return G_IO_STATUS_ERROR;
    }
  else
    {
      z_leave();
      return G_IO_STATUS_NORMAL;
    }
}

/**
 * z_disconnect:
 * @fd: socket to disconnect
 *
 * Disconnect an already connected socket for protocols that support this
 * operation. (for example UDP)
 **/
GIOStatus
z_disconnect(int fd)
{
  gint rc;
  struct sockaddr sa;

  z_enter();
  sa.sa_family = AF_UNSPEC;
  
  do
    {
      rc = connect(fd, &sa, sizeof(struct sockaddr));
    }
  while (rc == -1 && errno == EINTR);
  if (rc == -1)
    {
      z_log(NULL, CORE_ERROR, 3, "Disconnect failed; error='%m'");
      z_leave();
      return G_IO_STATUS_ERROR;
    }
  z_leave();
  return G_IO_STATUS_NORMAL;
}

/**
 * z_listen:
 * @fd: socket to listen
 * @backlog: the number of possible connections in the backlog queue
 * @accept_one: whether only one connection is to be accepted (used as a hint)
 *
 * Start listening on this socket given the underlying protocol supports it.
 **/
GIOStatus
z_listen(gint fd, gint backlog, gboolean accept_one)
{
  if (z_ll_listen(fd, backlog, accept_one) == -1)
    {
      /*LOG
        This message indicates that the listen() system call failed for
        the given fd.
       */
      z_log(NULL, CORE_ERROR, 3, "listen() failed; fd='%d', error='%m'", fd);
      return G_IO_STATUS_ERROR;
    }
  return G_IO_STATUS_NORMAL;
}

/**
 * z_getsockname:
 * @fd: socket
 * @local_addr: the local address is returned here
 *
 * Get the local address where a given socket is bound.
 **/ 
GIOStatus
z_getsockname(gint fd, ZSockAddr **local_addr)
{
  char sabuf[1500];
  socklen_t salen = sizeof(sabuf);
  
  if (z_ll_getsockname(fd, (struct sockaddr *) sabuf, &salen) == -1)
    {
      /*LOG
        This message indicates that the getsockname() system call failed
        for the given fd.
       */
      z_log(NULL, CORE_ERROR, 3, "getsockname() failed; fd='%d', error='%m'", fd);
      return G_IO_STATUS_ERROR;
    }
  *local_addr = z_sockaddr_new((struct sockaddr *) sabuf, salen);
  return G_IO_STATUS_NORMAL;
}

/**
 * z_getpeername:
 * @fd: socket
 * @peer_addr: the address of the peer is returned here
 *
 * Get the remote address where a given socket is connected.
 **/ 
GIOStatus
z_getpeername(gint fd, ZSockAddr **peer_addr)
{
  char sabuf[1500];
  socklen_t salen = sizeof(sabuf);
  
  if (z_ll_getpeername(fd, (struct sockaddr *) sabuf, &salen) == -1)
    {
      return G_IO_STATUS_ERROR;
    }
  *peer_addr = z_sockaddr_new((struct sockaddr *) sabuf, salen);
  return G_IO_STATUS_NORMAL;
}

/**
 * z_getdestname:
 * @fd: socket
 * @peer_addr: the address of the peer's original destination is returned here
 *
 * Get the original destination of a client represented by the socket.
 **/ 
GIOStatus
z_getdestname(gint fd, ZSockAddr **dest_addr)
{
  char sabuf[1500];
  socklen_t salen = sizeof(sabuf);
  
  if (z_ll_getdestname(fd, (struct sockaddr *) sabuf, &salen) == -1)
    {
      return G_IO_STATUS_ERROR;
    }
  *dest_addr = z_sockaddr_new((struct sockaddr *) sabuf, salen);
  return G_IO_STATUS_NORMAL;
}

/* low level functions */
gint
z_do_ll_bind(int fd, struct sockaddr *sa, socklen_t salen)
{
  return bind(fd, sa, salen);
}

gint
z_do_ll_accept(int fd, struct sockaddr *sa, socklen_t *salen)
{
  return accept(fd, sa, salen);
}

gint 
z_do_ll_connect(int fd, struct sockaddr *sa, socklen_t salen)
{
  return connect(fd, sa, salen);
}

gint 
z_do_ll_listen(int fd, gint backlog, gboolean accept_one)
{
  return listen(fd, backlog);
}

gint
z_do_ll_getsockname(int fd, struct sockaddr *sa, socklen_t *salen)
{
  return getsockname(fd, sa, salen);
}

gint 
z_do_ll_getpeername(int fd, struct sockaddr *sa, socklen_t *salen)
{
  return getpeername(fd, sa, salen);
}

ZSocketFuncs z_socket_funcs = 
{
  z_do_ll_bind,
  z_do_ll_accept,
  z_do_ll_connect,
  z_do_ll_listen,
  z_do_ll_getsockname,
  z_do_ll_getpeername,
  z_do_ll_getsockname
};

ZSocketFuncs *socket_funcs = &z_socket_funcs;

#if ZORPLIB_ENABLE_IPOPTIONS

static gboolean
z_get_ip_options(ZSocket *s, ZSockAddr *a)
{
  guchar buf[128]; /* ip options can be up to 40 bytes in size */
  guint buflen; 
  
  if (a->sa.sa_family != AF_INET)
    /* only AF_INET is supported */
    return FALSE;

  buflen = sizeof(buf);
  if (getsockopt(fd, SOL_IP, IP_OPTIONS, &buf, &buflen) == 0)
    {
      guchar newbuf[128];
      guchar *src = buf, *dst = newbuf;
      guint left = buflen, length;
      
      while (left)
        {
          if (*src == '\0')
            {
              /* EOOL */
              if (left > 1)
                {
                  /* hmm... this seems to be an invalid option list to me */
                  /*LOG
                    This message indicates that processing IP options failed,
                    because data was found after end-of-option-list. This might
                    be the result of a buggy TCP stack or an attack attempt.
                   */
                  z_log(NULL, CORE_ERROR, 1, "Invalid IP options, data after EOOL;");
                  return FALSE;
                }
              left--;
              break;
            }
          else if (*src == '\1')
            {
              /* NOP */
              left--;
            }
          else if (*src == 130 || *src == 133 || *src == 134)
            {
              /* RIPSO and CIPSO */
              
              /* this option is copied if present */
              if (left <= 1)
                {
                  /* no space for length field */
                  return FALSE;
                }
              length = *(src + 1);
              if (length > left)
                {
                  /* option too long */
                  return FALSE;
                }
              memcpy(dst, src, length);
              left -= length;
              dst += length;
              src += length;
            }
          else 
            {
              /* everything else is skipped */
              if (left <= 1)
                return FALSE;
              length = *(src + 1);
              left -= length;
              dst += length;
              src += length;
            }
        }
      z_sockaddr_inet_set_options(a, newbuf, dst - newbuf);
    }
  return TRUE;
}

static gboolean
z_set_ip_options(ZSocket *s, ZSockAddr *a)
{
  gpointer options = NULL;
  guint options_length;
    
  if (a->sa.sa_family != AF_INET)
    return FALSE;
  z_sockaddr_inet_get_options(a, &options, &options_length);
  if (options)
    {
      if (setsockopt(fd, SOL_IP, IP_OPTIONS, options, options_length) < 0)
        return FALSE;
    }
  return TRUE;    
}

#endif

#ifdef G_OS_WIN32

#include <winsock2.h>

gboolean
z_socket_init(void)
{
  WSADATA wsa;
  int res;

  res = WSAStartup(MAKEWORD(2,0), &wsa);
  if (res)
    {
      /*LOG
        This message attempt when Winsock startup failed in
        Windows Sysytems.
       */
      z_log(NULL, CORE_DEBUG, 0, "WinSock startup failed, error code %d", WSAGetLastError() );
      return FALSE;
    }
  return TRUE;
}

void
z_socket_done(void)
{
  WSACleanup();
}

#else

gboolean
z_socket_init(void)
{
  return TRUE;
}

void
z_socket_done(void)
{
}

#endif
